/**
* \file: WaylandTouchInput.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Digital iPod Out - Wayland Adapter
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <string.h>
#include <adit_logging.h>
#include <dipo_macros.h>
#include "WaylandTouchInput.h"
#include "HIDDigitizer.h"
#include "TouchListener.h"
#include "PointerListener.h"

LOG_IMPORT_CONTEXT(dipo_wl);

namespace adit { namespace carplay
{

using namespace std;

enum WaylandDeviceType
{
    AutoDetectType        = 1,
    PointerDeviceType     = 2,
    SingleTouchDeviceType = 3,
    MultiTouchDeviceType  = 4
};

// currently touch only supports 1 session, in case of overlap kill the old one
WaylandTouchInput* WaylandTouchInput::current = nullptr;

WaylandTouchInput::WaylandTouchInput()
{
    digitizer = nullptr;
    receiver = nullptr;
    surface = nullptr;
    wayland = nullptr;

    if (current != nullptr)
    {
        LOG_ERROR((dipo_wl, "old touch session still active; force exit"));
        current->release();
    }
    current = this;
}

WaylandTouchInput::~WaylandTouchInput()
{
    release();
    current = nullptr;
}

bool WaylandTouchInput::Initialize(const IConfiguration& inConfig, IInputReceiver& inReceiver, SessionId inSessionId)
{
    (void)inSessionId;

    receiver = &inReceiver;
    bool multiTouch = 0; // set multitouch to FALSE

    // get configuration
    int layerId = inConfig.GetNumber("wl-touch-layer-id", 3001);
    int surfaceId = inConfig.GetNumber("wl-touch-surface-id", 41);
    int width = inConfig.GetNumber("display-width", 800);
    int height = inConfig.GetNumber("display-height", 480);
    string displayUuid = inConfig.GetItem("display-uuid", "");
    string touchUuid = inConfig.GetItem("wl-touch-uuid", "");

    int vendorId = inConfig.GetNumber("wl-touch-hid-vendor-id", 0);
    int productId = inConfig.GetNumber("wl-touch-hid-product-id", 0);
    int countryCode = inConfig.GetNumber("wl-touch-hid-country-code", 0);

    string verboseConfig;
    bool verbose = false;
    if (inConfig.TryGetItem("enable-verbose-logging", verboseConfig))
    {
        verbose = 0 != inConfig.GetNumber("enable-verbose-logging", 0);
        LOG_WARN((dipo_wl, "verbose logging enabled, do not use in production!"));
    }

    // check touch-type and interval-usec, all other are already items checked somewhere else
    int deviceType = inConfig.GetNumber("wl-touch-type", 1);
    if (deviceType <= 0 || deviceType > 4)
    {
        LOG_WARN((dipo_wl, "wl-touch-type: invalid configuration value. set default to 1 (auto)"));
        deviceType = AutoDetectType;
    }

    if (deviceType == AutoDetectType || deviceType == MultiTouchDeviceType)
    {
	LOG_WARN((dipo_wl, "MultiTouch not supported in CarPlay. Change wl-touch-type in cfg file "));
    	multiTouch = 1; // enable multitouch
    }

    int usec = inConfig.GetNumber("wl-touch-input-trigger-interval-usec", 50);
    if (usec <= 0)
    {
        usec = 50;
        LOG_WARN((dipo_wl, "invalid time interval for input, set default to %d usec", usec));
    }

    // create HID pointer device utils
    digitizer = new HIDDigitizer(touchUuid, displayUuid, "Wayland Touchscreen Device",
    		multiTouch/* PointerDeviceType*/ /* TODO ADIT */, vendorId, productId, countryCode);

    // create wayland
    wayland = new WaylandContext();
    dipo_return_value_on_null(dipo, wayland, false);
    if (deviceType == AutoDetectType || deviceType == SingleTouchDeviceType ||
    	deviceType == MultiTouchDeviceType)
    {
        // add touch device listener
        unique_ptr<WaylandContext::DeviceListener> touch(
                new TouchListener(inReceiver, *digitizer, width, height, verbose));
        wayland->SetTouchListener(&TouchListener::Listener, move(touch));
    }

    if (deviceType == AutoDetectType || deviceType == PointerDeviceType)
    {
        // add pointer device listener
        unique_ptr<WaylandContext::DeviceListener> pointer(
                new PointerListener(inReceiver, *digitizer, width, height, verbose));
        wayland->SetPointerListener(&PointerListener::Listener, move(pointer));
    }

    receiver->AttachInput(digitizer->AsHIDDevice());

    // init
    if (!wayland->Initialize())
    {
        // error logged
        return false;
    }

    // create surface
    surface = new WaylandSurface();
    dipo_return_value_on_null(dipo, surface, false);
    if (!surface->Create(wayland, width, height, layerId, surfaceId))
    {
        // error logged
        return false;
    }

    // run input handler thread
    if (!wayland->StartInputThread(usec))
    {
        // error logged
        return false;
    }

    return true;
}

void WaylandTouchInput::release()
{
    dipo_safe_delete(surface);
    dipo_safe_delete(wayland);
    if (digitizer != nullptr && receiver != nullptr)
        receiver->DetachInput(digitizer->AsUUID());
    dipo_safe_delete(digitizer);
}

} } // namespace adit { namespace carplay
